<?php
/**
 *
 * Let's Review Settings Class
 *
 * @since      1.0.0
 * @copyright   Copyright Codetipi
 *
 * @package    Let's Review
 * @subpackage lets-review/admin
 */

class Lets_Review_Settings {

	/**
	 * Var for Let's Review slug.
	 *
	 * @since    1.0.0
	 */
	private $slug;

	/**
	 * Var for Let's Review version.
	 *
	 * @since    1.0.0
	 */
	private $version;

	/**
	 * Var for Let's Review URL.
	 *
	 * @since    1.0.0
	 */
	private $url;

	/**
	 * Var for Let's Review name.
	 *
	 * @since    1.0.0
	 */
	private $name;

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	*/
	public function __construct( $slug = '', $version = '', $url = '', $name = '' ) {

		$this->url     = $url;
		$this->version = $version;
		$this->name    = $name;
		$this->slug    = $slug;

		add_action( 'admin_init', array( $this, 'lets_review_settings' ) );
		add_action( 'admin_menu', array( $this, 'lets_review_menu_page' ) );
	}

	/**
	 * Admin Menu Page
	 *
	 * @since 1.0.0
	 */
	public function lets_review_menu_page() {
		add_menu_page( "Let's Review", "Let's Review", 'manage_options', $this->slug, array( $this, 'lets_review_page_loader' ), esc_url( $this->url . 'admin/images/lets-review-icon.png' ) );
		add_submenu_page( $this->slug, esc_html__( 'Welcome', 'lets-review' ), esc_html__( 'Welcome', 'lets-review' ), 'manage_options', $this->slug, array( $this, 'lets_review_page_loader' ) );
		add_submenu_page( $this->slug, esc_html__( 'Options', 'lets-review' ), esc_html__( 'Options', 'lets-review' ), 'manage_options', $this->slug . '-general', array( $this, 'lets_review_page_loader' ) );
		add_submenu_page( $this->slug, esc_html__( 'Default Values', 'lets-review' ), esc_html__( 'Default Values', 'lets-review' ), 'manage_options', $this->slug . '-default', array( $this, 'lets_review_page_loader' ) );
		add_submenu_page( $this->slug, esc_html__( 'Help', 'lets-review' ), esc_html__( 'Help', 'lets-review' ), 'manage_options', $this->slug . '-help', array( $this, 'lets_review_page_loader' ) );
		add_submenu_page( $this->slug, esc_html__( 'Migration', 'lets-review' ), esc_html__( 'Migration', 'lets-review' ), 'manage_options', $this->slug . '-migration', array( $this, 'lets_review_page_loader' ) );

	}

	/**
	 * Page Loader
	 *
	 * @version 2.0.0
	 * @since 1.0.0
	 */
	public function lets_review_page_loader() {

		global $plugin_page;
		$this->lets_review_admin_head();
		switch ( $plugin_page ) {

			case $this->slug . '-general':
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/pages/general.php';
				break;
			case $this->slug . '-default':
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/pages/default.php';
				break;
			case $this->slug . '-help':
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/pages/help.php';
				break;
			case $this->slug . '-migration':
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/pages/migration.php';
				break;
			default:
				require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/pages/welcome.php';
				break;
		}

		$this->lets_review_admin_footer();

	}

	/**
	 * Page Settings
	 * @version 2.0.0
	 * @since 1.0.0
	 */
	public function lets_review_settings() {

		add_settings_section(
			'lets_review_set_section_general',
			'',
			'',
			'lets-review-general'
		);

		add_settings_field(
			'lets_review_schema_onoff',
			esc_html__( 'JSON-LD Review Schema', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'    => 'lets_review_schema_onoff',
				'default' => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_lb',
			esc_html__( 'Load  Let\'s Review Lightbox', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'    => 'lets_review_gen_lb',
				'default' => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_lb_custom_class',
			esc_html__( 'Custom class for external lightbox', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_text' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'lets_review_gen_lb_custom_class',
				'cb-custom-shopping-icon',
				'cb-hidden-load',
				'required'       => 'lets_review_gen_lb',
				'required_value' => 'off',
			)
		);

		add_settings_field(
			'lets_review_gen_fa',
			esc_html__( 'Load FontAwesome files', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'    => 'lets_review_gen_fa',
				'default' => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_js_min',
			esc_html__( 'Use Minified Javascript', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'    => 'lets_review_gen_js_min',
				'default' => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_load_outside_post',
			esc_html__( 'Show full review outside post', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'        => 'lets_review_gen_load_outside_post',
				'description' => esc_html__( 'To control reviews showing in themes that show entire post content on homepage/categories/etc.', 'lets-review' ),
				'default'     => 'on',
			)
		);

		add_settings_field(
			'lets_review_stars_f_score_w_number',
			esc_html__( 'Always show final score number', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'        => 'lets_review_stars_f_score_w_number',
				'description' => esc_html__( 'Enable this to show the final score number above the stars/icons/image in final score boxes.', 'lets-review' ),
				'default'     => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_type_shopping_onoff',
			esc_html__( 'Change shopping cart icon', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'    => 'lets_review_gen_type_shopping_onoff',
				'default' => 'off',
			)
		);

		add_settings_field(
			'lets_review_gen_type_shopping_icon',
			esc_html__( 'Shopping Cart Icon', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_text' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'lets_review_gen_type_shopping_icon',
				'cb-custom-shopping-icon',
				'cb-hidden-load',
				'required'       => 'lets_review_gen_type_shopping_onoff',
				'required_value' => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_rating_permission',
			esc_html__( 'Who can leave User Rating', 'lets-review' ),
			array( $this, 'lets_review_settings_select' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name' => 'lets_review_gen_rating_permission',
				'args' => array(
					1 => esc_html__( 'Everyone', 'lets-review' ),
					2 => esc_html__( 'Only logged in users', 'lets-review' ),
				),
			)
		);

		add_settings_field(
			'lets_review_gd_reader_text',
			esc_html__( 'Visitor rating title', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_text' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'lets_review_gd_reader_text',
				'default' => esc_html__( 'Reader Rating', 'lets-review' ),
			)
		);

		add_settings_field(
			'lets_review_gen_user_reviews',
			esc_html__( 'Add Review Option To Comments', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'        => 'lets_review_gen_user_reviews',
				'description' => esc_html__( 'Adds the posts review fields to the comment form for visitors to leave reviews. Only applies to posts with reviews enabled.', 'lets-review' ),
				'default'     => 'off',
			)
		);

		add_settings_field(
			'lets_review_comments_up_down',
			esc_html__( 'Up Down Votes in Comment Reviews', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'        => 'lets_review_comments_up_down',
				'description' => '',
				'default'     => '1',
			)
		);
		add_settings_field(
			'lets_review_comments_normal_up_down',
			esc_html__( 'Up Down Votes in Normal Comments', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'        => 'lets_review_comments_normal_up_down',
				'description' => '',
				'default'     => '',
			)
		);

		add_settings_field(
			'lets_review_gen_user_review_onoff_default',
			esc_html__( 'Default "Is this a review?" value', 'lets-review' ),
			array( $this, 'lets_review_settings_select' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name'           => 'lets_review_gen_user_review_onoff_default',
				'args'           => array(
					1 => esc_html__( 'Off', 'lets-review' ),
					2 => esc_html__( 'On', 'lets-review' ),
				),
				'description'    => esc_html__( 'If you want to really push users to leave a review, you can set the "Is this a review?" option in the comment form to be set to "On" by default in new user comments.', 'lets-review' ),
				'required'       => 'lets_review_gen_user_reviews',
				'required_value' => 'on',
			)
		);

		add_settings_field(
			'lets_review_gen_type_headings',
			esc_html__( 'Font: Headings', 'lets-review' ),
			array( $this, 'lets_review_settings_select' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name' => 'lets_review_gen_type_headings',
				'args' => $this->lets_review_fonts(),
			)
		);

		add_settings_field(
			'lets_review_gen_type_p',
			esc_html__( 'Font: Parragraphs', 'lets-review' ),
			array( $this, 'lets_review_settings_select' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name' => 'lets_review_gen_type_p',
				'args' => $this->lets_review_fonts(),
			)
		);

		add_settings_field(
			'lets_review_gen_type_ext',
			esc_html__( 'Font: Character Sets', 'lets-review' ),
			array( $this, 'lets_review_settings_multiselect' ),
			'lets-review-general',
			'lets_review_set_section_general',
			array(
				'name' => 'lets_review_gen_type_ext',
				'args' => array(
					'latin'        => 'Latin',
					'latin-ext'    => 'Latin Extended',
					'cyrillic'     => 'Cyrillic',
					'cyrillic-ext' => 'Cyrillic Extended',
					'vietnamese'   => 'Vietnamese',
					'greek'        => 'Greek',
					'greek-ext'    => 'Greek Extended',
				),
			)
		);

		add_settings_section(
			'lets_review_set_section_migration',
			'',
			'',
			'lets-review-migration'
		);

		add_settings_field(
			'lets_review_migration_source',
			esc_html__( 'Old Review System', 'lets-review' ),
			array( $this, 'lets_review_settings_migration' ),
			'lets-review-migration',
			'lets_review_set_section_migration',
			array(
				'lets_review_migration_source',
			)
		);

		add_settings_section(
			'lets_review_set_section_defaults',
			'',
			'',
			'lets-review-defaults'
		);

		add_settings_field(
			'lets_review_gd_format',
			esc_html__( 'Format', 'lets-review' ),
			array( $this, 'lets_review_settings_radio_img' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_format',
				array( 'cb-review-format cb-trigger-wrap', 'cb-custom-icon', 4 ),
				array(
					array( esc_html__( 'Percent', 'lets-review' ), 'lr-percent.png' ),
					array( esc_html__( 'Points', 'lets-review' ), 'lr-points.png' ),
					array( esc_html__( 'Stars', 'lets-review' ), 'lr-stars.png' ),
					array( esc_html__( 'Custom Icon', 'lets-review' ), 'lr-custom-icon.png' ),
					array( esc_html__( 'Custom Image', 'lets-review' ), 'lr-custom-image.png' ),
				),

			)
		);

		add_settings_field(
			'lets_review_gd_custom_icon',
			esc_html__( 'Custom Icon Code', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_text' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_custom_icon',
				'cb-custom-icon',
				'cb-hidden-load',
			)
		);

		add_settings_field(
			'lets_review_gd_custom_image',
			esc_html__( 'Custom Icon Image', 'lets-review' ),
			array( $this, 'lets_review_settings_media' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'name' => 'lets_review_gd_custom_image',
			)
		);

		add_settings_field(
			'lets_review_gd_location',
			esc_html__( 'Location', 'lets-review' ),
			array( $this, 'lets_review_settings_radio_img' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_location',
				'cb-review-location',
				array(
					array( esc_html__( 'Bottom', 'lets-review' ), 'lr-bot.png' ),
					array( esc_html__( 'Top', 'lets-review' ), 'lr-top.png' ),
					array( esc_html__( 'Top + Bottom', 'lets-review' ), 'lr-top-bot.png' ),
					array( esc_html__( 'Shortcode', 'lets-review' ), 'lr-shortcode.png' ),
				),
			)
		);

		add_settings_field(
			'lets_review_gd_design',
			esc_html__( 'Design', 'lets-review' ),
			array( $this, 'lets_review_settings_radio_img' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_design',
				array( 'cb-review-design cb-trigger-wrap', 'cb-review-animation cb-review-skin', array( 3, 4, 5 ) ),
				array(
					array( esc_html__( 'Minimalist', 'lets-review' ), 'lr-skin-1.png' ),
					array( esc_html__( 'Bold', 'lets-review' ), 'lr-skin-2.png' ),
					array( esc_html__( 'Modern', 'lets-review' ), 'lr-skin-3.png' ),
					array( esc_html__( 'Modern B', 'lets-review' ), 'lr-skin-4.png' ),
					array( esc_html__( 'Clean', 'lets-review' ), 'lr-skin-5.png' ),
					array( esc_html__( 'Minimalist B', 'lets-review' ), 'lr-skin-6.png' ),
				),
			)
		);

		add_settings_field(
			'lets_review_gd_ani',
			esc_html__( 'Animation Effect', 'lets-review' ),
			array( $this, 'lets_review_settings_radio_img' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_ani',
				'cb-review-animation',
				array(
					array( esc_html__( 'Incremental', 'lets-review' ), 'lr-ani-1.gif' ),
					array( esc_html__( 'Fade In', 'lets-review' ), 'lr-ani-2.gif' ),
					array( esc_html__( 'None', 'lets-review' ), 'lr-ani-off.png' ),
				),
			)
		);

		add_settings_field(
			'lets_review_gd_skin',
			esc_html__( 'Skin Style', 'lets-review' ),
			array( $this, 'lets_review_settings_radio_img' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_skin',
				'cb-review-skin',
				array(
					array( esc_html__( 'Light', 'lets-review' ), 'lr-light.png' ),
					array( esc_html__( 'Dark', 'lets-review' ), 'lr-dark.png' ),
				),
			)
		);

		add_settings_field(
			'lets_review_gd_color',
			esc_html__( 'Accent Color', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_color' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_color',
				'cb-review-color',
			)
		);

		add_settings_field(
			'lets_review_gd_title_use_post_title',
			esc_html__( 'Use Post Title for Review Title', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'name'    => 'lets_review_gd_title_use_post_title',
				'default' => 'off',
			)
		);

		add_settings_field(
			'lets_review_gd_media_use_fi',
			esc_html__( 'Use Featured Image as Main Review Image', 'lets-review' ),
			array( $this, 'lets_review_settings_switch' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'name'    => 'lets_review_gd_media_use_fi',
				'default' => 'off',
			)
		);

		add_settings_field(
			'lets_review_gd_title_score_subtitle',
			esc_html__( 'Score Subtitle', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_title' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_title_score_subtitle',
			)
		);

		add_settings_field(
			'lets_review_gd_title_pros',
			esc_html__( 'Positives Title', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_title' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_title_pros',
			)
		);

		add_settings_field(
			'lets_review_gd_title_cons',
			esc_html__( 'Negatives Title', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_title' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_title_cons',
			)
		);

		add_settings_field(
			'lets_review_gd_title_aff',
			esc_html__( 'Affiliate Block Title', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_title' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_title_aff',
			)
		);

		add_settings_field(
			'lets_review_gd_title_conclusion',
			esc_html__( 'Conclusion Title', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_title' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_title_conclusion',
			)
		);

		add_settings_field(
			'lets_review_gd_title_gallery',
			esc_html__( 'Gallery Block Title', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_title' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_gd_title_gallery',
			)
		);

		add_settings_field(
			'lets_review_criterias',
			esc_html__( 'Criterias', 'lets-review' ),
			array( $this, 'lets_review_settings_cb_crits' ),
			'lets-review-defaults',
			'lets_review_set_section_defaults',
			array(
				'lets_review_criterias',
				'cb-criteria-block',
			)
		);

		register_setting( 'lets-review-settings-migration', 'lets_review_migration_source', array( $this, 'lets_review_migration_cb' ) );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_fa', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_js_min', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_load_outside_post', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_stars_f_score_w_number', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_lb', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_schema_onoff', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_type_ext', array( $this, 'lets_review_array_sanitize' ) );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_type_p', 'esc_attr' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_lb_custom_class', 'wp_kses_data' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_type_shopping_icon', '' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_type_shopping_onoff', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_user_reviews', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_comments_up_down', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_comments_normal_up_down', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_user_review_onoff_default', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_type_headings', 'esc_attr' );
		register_setting( 'lets-review-settings-general', 'lets_review_gen_rating_permission', 'intval' );
		register_setting( 'lets-review-settings-general', 'lets_review_gd_reader_text', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_use_post_title', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_media_use_fi', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_skin', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_conclusion', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_score_subtitle', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_pros', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_cons', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_aff', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_title_gallery', 'sanitize_text_field' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_color', 'esc_html' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_custom_icon', 'lets_review_sanitize_wp_kses' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_custom_image', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_location', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_format', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_criterias', array( $this, 'lets_review_criterias_sanitize' ) );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_ani', 'intval' );
		register_setting( 'lets-review-settings-gd', 'lets_review_gd_design', 'intval' );

	}

	public function lets_review_settings_cb_text( $args ) {
		$current           = get_option( $args[0] );
		$current           = empty( $current ) && ! empty( $args['default'] ) ? $args['default'] : $current;
		$lets_review_class = '';
		$lets_review_id    = '';
		if ( isset( $args[1] ) ) {
			$lets_review_id = $args[1];
		}

		if ( isset( $args[2] ) ) {
			$lets_review_class = $args[2];
		}
		$req = $req_value = '';
		if ( ! empty( $args['required'] ) ) {
			$req       = $args['required'];
			$req_value = $args['required_value'];
		}

		?>
		<div id="<?php echo esc_attr( $lets_review_id ); ?>" data-required="<?php echo esc_attr( $req ); ?>" data-required-val="<?php echo esc_attr( $req_value ); ?>" class="cb-element cb-text <?php echo esc_attr( $lets_review_class ); ?> <?php echo esc_attr( $lets_review_id ); ?> lr-cf">
			<input type="text" id="<?php echo esc_attr( $args[0] ); ?>" name="<?php echo esc_attr( $args[0] ); ?>" value="<?php echo esc_html( $current ); ?>">
		</div>
		<?php
	}

	public function lets_review_settings_media( $args ) {
		$current = get_option( $args['name'] );
		?>

		<div id="cb-custom-image-wrap" class="cb-trigger-block cb-media-cont lr-cf">
			<div id="cb-custom-image" class="lr-cf">
				<div class="cb-option lr-cf cb-mb">
					<span class="cb-list-title"><?php esc_html_e( 'Custom Image', 'lets-review' ); ?></span>
					<div class="cb-desc"><?php esc_html_e( 'Upload/Set an image of a single object. It will output 5 times automatically. Recommended size is 80px x 80px.', 'lets-review' ); ?></div>
				</div>

				<?php if ( ! empty( $current ) ) { ?>

					<?php $lets_review_custom_image_img = wp_get_attachment_image_src( $current, 'thumbnail' ); ?>

						<span data-cb-id="<?php echo intval( $current ); ?>" class="cb-gallery-img">
							<a href="#" class="cb-button cb-remove">&#xd7;</a>
							<img src="<?php echo esc_url( $lets_review_custom_image_img[0] ); ?>" alt="">
							<input type="hidden" value="<?php echo intval( $current ); ?>" name="<?php echo esc_attr( $args['name'] ); ?>">
						</span>

				<?php } ?>

			</div>

			<a href="#" class="cb-single-image-trigger cb-button" data-cb-dest="cb-custom-image" data-cb-name="<?php echo esc_attr( $args['name'] ); ?>"><?php esc_html_e( 'Select Image', 'lets-review' ); ?></a>
		</div>
		<?php
	}

	public function lets_review_settings_switch( $args ) {

		$default = ! empty( $args['default'] ) && $args['default'] == 'on' ? 1 : 0;
		$current = get_option( $args['name'], $default );

		?>
		<input type="checkbox" value="1" id="<?php echo esc_attr( $args['name'] ); ?>" class="lets-review-switch lets-review-switch-s" name="<?php echo esc_attr( $args['name'] ); ?>" <?php checked( 1, $current ); ?>>
		<?php
		if ( ! empty( $args['description'] ) ) {
			echo '<p class="description">' . esc_html( $args['description'] ) . '</p>'; }
	}

	public function lets_review_settings_migration( $args ) {
		?>
		<select id="lets-review-migration" name="lets_review_migration" class="cb-select">
			<option value="1" selected="selected">15Zine Reviews</option>
			<option value="2">Valenti Reviews</option>
			<option value="4">Ciola Reviews</option>
			<option value="3">WP Product Review (Themeisle) Reviews</option>
			<option value="5">Extra Theme/Divi Reviews (Elegant Themes)</option>
		</select>
		<?php
	}

	public function lets_review_settings_select( $args ) {

		$current    = empty( $args['name'] ) ? '' : get_option( $args['name'] );
		 $req       = '';
		 $req_value = '';
		if ( ! empty( $args['required'] ) ) {
			$req       = $args['required'];
			$req_value = $args['required_value'];
		}
		?>
		<select id="<?php echo esc_attr( $args['name'] ); ?>" class="cb-element" name="<?php echo esc_attr( $args['name'] ); ?>" data-required="<?php echo esc_attr( $req ); ?>" data-required-val="<?php echo esc_attr( $req_value ); ?>">
			<?php foreach ( $args['args'] as $value => $name ) { ?>
				<option value="<?php echo esc_attr( $value ); ?>" <?php selected( $value, $current ); ?>><?php echo esc_attr( $name ); ?></option>
			<?php } ?>
		</select>
		<?php
		if ( ! empty( $args['description'] ) ) {
			echo '<p class="description">' . esc_html( $args['description'] ) . '</p>';
		}
	}

	public function lets_review_settings_multiselect( $args ) {

		$current = empty( $current ) ? array() : get_option( $args['name'] );
		?>
		<select multiple="multiple" id="cb-cat-<?php echo esc_attr( $args['name'] ); ?>"  name="<?php echo esc_attr( $args['name'] ); ?>[]" class="cb-select cb-select-sol">

			<?php foreach ( $args['args'] as $value => $name ) { ?>
				<option value="<?php echo esc_attr( $value ); ?>" <?php if ( in_array( $value, $current ) ) { ?>selected="selected"<?php } ?>><?php echo esc_attr( $name ); ?></option>
			<?php } ?>

		</select>

		<?php
	}

	public function lets_review_fonts() {

		return apply_filters(
			'lets_review_fonts',
			array(
				'none'         => esc_html__( 'Inherit from Theme', 'lets-review' ),
				'Arvo'         => 'Arvo',
				'Lobster'      => 'Lobster',
				'Merriweather' => 'Merriweather',
				'Montserrat'   => 'Montserrat',
				'Open Sans'    => 'Open Sans',
				'Oswald'       => 'Oswald',
				'Pacifico'     => 'Pacifico',
				'Quattrocento' => 'Quattrocento',
				'Raleway'      => 'Raleway',
				'Ubuntu'       => 'Ubuntu',
			)
		);

	}

	public function lets_review_settings_cb_crits( $args ) {
		$lets_review_criterias    = get_option( $args[0] );
		$lets_review_crit_counter = 1;
		?>

		<div class="cb-element lr-cf cb-criteria-block">
			<div class="cb-desc cb-mb"><?php esc_html_e( 'Add as many criterias as you like. These will appear in every new review. You will be able to remove or add more of them inside each post if required. You can drag & drop the items around here.', 'lets-review' ); ?></div>
			<ul id="cb-criterias">
				<?php if ( isset( $lets_review_criterias ) && is_array( $lets_review_criterias ) ) { ?>
					<?php foreach ( $lets_review_criterias as $lets_review_crit ) { ?>

						<li id="cb-criteria-<?php echo intval( $lets_review_crit_counter ); ?>" class="ui-state-default cb-list-field cb-criteria-field lr-cf" data-cb-score="<?php if ( isset( $lets_review_crit['score'] ) ) { echo intval( $lets_review_crit['score'] ); } ?>" data-cb-crit-title="<?php if ( isset( $lets_review_crit['title'] ) ) { echo esc_html( $lets_review_crit['title'] ); } ?>">
							<div class="cb-criteria-title">
								<span class="cb-list-title"><?php esc_html_e( 'Title', 'lets-review' ); ?></span>
								<input type="text" value="<?php if ( isset( $lets_review_crit['title'] ) ) { echo esc_html( $lets_review_crit['title'] ); } ?>" id="cb-criteria-field-<?php echo intval( $lets_review_crit_counter ); ?>" name="lets_review_criterias[<?php echo intval( $lets_review_crit_counter ); ?>][title]" class="cb-input">
							</div>
							<div class="cb-criteria-score lr-cf">
								<span class="cb-list-title"><?php esc_html_e( 'Score', 'lets-review' ); ?></span>
								<div class="cb-review-slider cb-slider"></div>
								<input type="text" value="<?php if ( isset( $lets_review_crit['score'] ) ) { echo intval( $lets_review_crit['score'] ); } ?>" name="lets_review_criterias[<?php echo intval( $lets_review_crit_counter ); ?>][score]" class="cb-cri-score cb-input" readonly>
							</div>
							<a href="#" class="cb-button cb-remove">&#xd7;</a>
						</li>
						<?php $lets_review_crit_counter++; ?>

					<?php } ?>
				<?php } ?>

			</ul>
		</div>

		<a href="#" class="cb-button cb-add-new cb-add-criteria" data-target="cb-criterias" data-cb-counter="<?php echo intval( $lets_review_crit_counter ); ?>"><?php esc_html_e( 'Add', 'lets-review' ); ?></a>

		<?php
	}

	public function lets_review_settings_cb_color( $args ) {
		$current = ( get_option( $args[0] ) == '' ) ? '#f8d92f' : get_option( $args[0] );
		?>
		<input type="text" class="lets-review-colorpicker" name="<?php echo esc_attr( $args[0] ); ?>" value="<?php echo esc_attr( $current ); ?>">
		<?php
	}

	public function lets_review_settings_cb_title( $args ) {
		$current = ( get_option( $args[0] ) == '' ) ? '' : get_option( $args[0] );
		?>
		<input type="text" class="lets-review-title" name="<?php echo esc_attr( $args[0] ); ?>" value="<?php echo esc_attr( $current ); ?>">
		<?php
		if ( ! empty( $args[1] ) ) {
			echo '<p class="description">' . esc_html( $args[1] ) . '</p>';
		}
	}

	public function lets_review_settings_radio_img( $args ) {
		$i       = 1;
		$current = get_option( $args[0] );

		if ( empty( $current ) ) {
			$current = 1;
		}

		if ( is_array( $args[1] ) ) {
			$lets_review_attr  = $args[1][0];
			$lets_review_count = $args[1][2];
			$lets_review_data  = $args[1][1];
		} else {
			$lets_review_attr  = $args[1];
			$lets_review_count = '';
			$lets_review_data  = '';
		}
		?>

		<div class="cb-element cb-radio-images <?php echo esc_attr( $lets_review_attr ); ?> lr-cf cb-text" data-cb-trigger="<?php echo esc_attr( $lets_review_data ); ?>">

			<?php foreach ( $args[2] as $arg ) { ?>
				<?php
				if ( ( is_array( $lets_review_count ) && in_array( $i, $lets_review_count ) ) || $lets_review_count == $i ) {
					$lets_review_data_trigger = 'cb-trig';
				} else {
					$lets_review_data_trigger = '';
				}
				?>
				<div class="cb-radio-images-element lr-cf" data-cb-trigger="<?php echo esc_attr( $lets_review_data_trigger ); ?>">
					<span class="cb-radio-title"><?php echo esc_html( $arg[0] ); ?></span>
					 <label>
						<input type="radio" class="cb-input-radio" name="<?php echo esc_attr( $args[0] ); ?>" value="<?php echo intval( $i ); ?>" <?php checked( $i, $current ); ?>>
						<img src="<?php echo esc_url( $this->url . 'admin/images/' . $arg[1] ); ?>" alt="" class="cb-radio-image">
					</label>
				</div>

				<?php $i++; ?>
			<?php } ?>

		</div>

		<?php
	}

	/**
	 * Let's Review Criterias Sanitize
	 *
	 * @since 1.0.0
	 */
	public function lets_review_criterias_sanitize( $input ) {

		if ( is_array( $input ) ) {
			$lets_review_output = array();
			foreach ( $input as $keys ) {
				$lets_review_arr = array();
				foreach ( $keys as $key => $value ) {
					$lets_review_arr[ $key ] = esc_attr( $value );
				}
				$lets_review_output[] = $lets_review_arr;
			}
		} else {
			$lets_review_output = '';
		}

		return $lets_review_output;
	}

	/**
	 * Let's Review Array Sanitize
	 *
	 * @since 1.0.0
	 */
	public function lets_review_array_sanitize( $input ) {

		if ( is_array( $input ) ) {
			return array_map( 'esc_attr', $input );
		} else {
			return esc_attr( $input );
		}
	}

	/**
	 * Let's Review Migration Callback
	 *
	 * @since 2.0.0
	 */
	public function lets_review_migration_cb() {
		if ( ! isset( $_POST['lets-review-migration-nonce'] ) || ! wp_verify_nonce( $_POST['lets-review-migration-nonce'], 'lets-review-migration' ) ) {
			die();
		}

		switch ( $_POST['lets_review_migration'] ) {
			case 1:
				$qry = new WP_Query(
					array(
						'posts_per_page' => -1,
						'fields'         => 'ids',
						'meta_key'       => 'cb_review_checkbox',
						'meta_value'     => 'on',
					)
				);

				if ( empty( $qry->posts ) ) {
					return;
				}

				foreach ( $qry->posts as $pid ) {

					update_post_meta( $pid, 'cb_review_checkbox', 'off', 'on' );
					update_post_meta( $pid, '_lets_review_onoff', 1, 0 );

					if ( get_post_meta( $pid, 'cb_review_title_user', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_title', get_post_meta( $pid, 'cb_review_title_user', true ) );
					}

					if ( get_post_meta( $pid, 'cb_pros_title', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_pros_title', get_post_meta( $pid, 'cb_pros_title', true ) );
					}

					if ( get_post_meta( $pid, 'cb_cons_title', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_cons_title', get_post_meta( $pid, 'cb_cons_title', true ) );
					}

					if ( get_post_meta( $pid, 'cb_rating_short_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_subtitle_external', get_post_meta( $pid, 'cb_rating_short_summary', true ) );
					}

					if ( get_post_meta( $pid, 'cb_rating_short_summary_in', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_subtitle', get_post_meta( $pid, 'cb_rating_short_summary_in', true ) );
					}

					if ( get_post_meta( $pid, 'cb_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_conclusion', get_post_meta( $pid, 'cb_summary', true ) );
					}

					switch ( get_post_meta( $pid, 'cb_user_score', true ) ) {
						case 'cb-both':
							$type = 1;
							break;
						case 'cb-readers':
							$type = 3;
							break;
						default:
							$type = 2;
							break;
					}

					update_post_meta( $pid, '_lets_review_type', $type );

					if ( get_post_meta( $pid, 'cb_placement', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_placement', true ) ) {
							case 'top':
								$location = 2;
								break;
							case 'top-half':
								$location = 5;
								break;
							default:
								$location = 1;
								break;
						}
						update_post_meta( $pid, '_lets_review_location', $location );
					}

					if ( get_post_meta( $pid, 'cb_votes', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_user_rating_vote_count', get_post_meta( $pid, 'cb_votes', true ) );
					}

					if ( get_post_meta( $pid, 'cb_user_score_output', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_user_rating', get_post_meta( $pid, 'cb_user_score_output', true ) );
					}

					$pros_new  = array();
					$cons_new  = array();
					$criterias = array();

					if ( get_post_meta( $pid, 'cb_pros', true ) != '' ) {
						foreach ( get_post_meta( $pid, 'cb_pros', true ) as $pro ) {
							$pros_new[] = array( 'positive' => $pro['title'] );
						}
					}

					if ( ! empty( $pros_new ) ) {
						update_post_meta( $pid, '_lets_review_pros', $pros_new );
					}

					if ( get_post_meta( $pid, 'cb_cons', true ) != '' ) {
						foreach ( get_post_meta( $pid, 'cb_cons', true ) as $con ) {
							$cons_new[] = array( 'negative' => $con['title'] );
						}
					}

					if ( ! empty( $cons_new ) ) {
						update_post_meta( $pid, '_lets_review_cons', $cons_new );
					}

					if ( get_post_meta( $pid, 'cb_review_crits', true ) != '' ) {
						foreach ( get_post_meta( $pid, 'cb_review_crits', true ) as $key ) {

							switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
								case 'stars':
									$score = number_format( ( $key['cb_cs'] / 20 ), 1 );
									break;
								case 'points':
									$score = $key['cb_cs'] / 10;
									break;
								default:
									$score = $key['cb_cs'];
									break;
							}
							$criterias[] = array(
								'title' => $key['title'],
								'score' => $score,
							);
						}

						if ( ! empty( $criterias ) ) {
							update_post_meta( $pid, '_lets_review_criterias', $criterias );
						}
					}

					switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
						case 'stars':
							update_post_meta( $pid, '_lets_review_format', 3 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', number_format( ( get_post_meta( $pid, 'cb_final_score_override', true ) / 20 ), 1 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', number_format( ( get_post_meta( $pid, 'cb_final_score', true ) / 20 ), 1 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
						case 'points':
							update_post_meta( $pid, '_lets_review_format', 2 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', ( get_post_meta( $pid, 'cb_final_score_override', true ) / 10 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', ( get_post_meta( $pid, 'cb_final_score', true ) / 10 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
						default:
							update_post_meta( $pid, '_lets_review_format', 1 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', get_post_meta( $pid, 'cb_final_score_override', true ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', get_post_meta( $pid, 'cb_final_score', true ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
					}
				}

				wp_reset_postdata();
				break;
			case 2:
				$qry = new WP_Query(
					array(
						'posts_per_page' => -1,
						'fields'         => 'ids',
						'meta_key'       => 'cb_review_checkbox',
						'meta_value'     => 1,
					)
				);

				if ( empty( $qry->posts ) ) {
					return;
				}

				foreach ( $qry->posts as $pid ) {

					update_post_meta( $pid, 'cb_review_checkbox', 0, 1 );
					update_post_meta( $pid, '_lets_review_onoff', 1, 0 );

					if ( get_post_meta( $pid, 'cb_pros_title', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_pros_title', get_post_meta( $pid, 'cb_pros_title', true ) );
					}

					if ( get_post_meta( $pid, 'cb_cons_title', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_cons_title', get_post_meta( $pid, 'cb_cons_title', true ) );
					}

					if ( get_post_meta( $pid, 'cb_rating_short_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_subtitle_external', get_post_meta( $pid, 'cb_rating_short_summary', true ) );
					}

					if ( get_post_meta( $pid, 'cb_rating_short_summary_in', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_subtitle', get_post_meta( $pid, 'cb_rating_short_summary_in', true ) );
					}

					if ( get_post_meta( $pid, 'cb_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_conclusion', get_post_meta( $pid, 'cb_summary', true ) );
					}

					switch ( get_post_meta( $pid, 'cb_user_score', true ) ) {
						case 'cb-both':
							$type = 1;
							break;
						case 'cb-readers':
							$type = 3;
							break;
						default:
							$type = 2;
							break;
					}

					update_post_meta( $pid, '_lets_review_type', $type );

					if ( get_post_meta( $pid, 'cb_placement', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_placement', true ) ) {
							case 'top':
								$location = 2;
								break;
							case 'top-half':
								$location = 5;
								break;
							default:
								$location = 1;
								break;
						}
						update_post_meta( $pid, '_lets_review_location', $location );
					}

					if ( get_post_meta( $pid, 'cb_votes', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_user_rating_vote_count', get_post_meta( $pid, 'cb_votes', true ) );
					}

					if ( get_post_meta( $pid, 'cb_user_score_output', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_user_rating', get_post_meta( $pid, 'cb_user_score_output', true ) );
					}

					$pros = array( 'cb_pro_1', 'cb_pro_2', 'cb_pro_3' );

					$pros_new = $cons_new = $criterias = array();
					foreach ( $pros as $key ) {
						$pro = get_post_meta( $pid, $key, true );
						if ( $pro != '' ) {
							$pros_new[] = array( 'positive' => $pro );
						}
					}
					if ( ! empty( $pros_new ) ) {
						update_post_meta( $pid, '_lets_review_pros', $pros_new );
					}

					$cons = array( 'cb_con_1', 'cb_con_2', 'cb_con_3' );
					foreach ( $cons as $key ) {
						$con = get_post_meta( $pid, $key, true );
						if ( $con != '' ) {
							$cons_new[] = array( 'negative' => $con );
						}
					}
					if ( ! empty( $cons_new ) ) {
						update_post_meta( $pid, '_lets_review_cons', $cons_new );
					}

					for ( $i = 1; $i < 7; $i++ ) {
						$title = get_post_meta( $pid, "cb_ct$i", true ) != '' ? get_post_meta( $pid, "cb_ct$i", true ) : '';
						$score = get_post_meta( $pid, "cb_cs$i", true ) != '' ? get_post_meta( $pid, "cb_cs$i", true ) : '';

						if ( get_post_meta( $pid, 'cb_score_display_type', true ) != '' ) {
							switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
								case 'stars':
									$score = number_format( ( $score / 20 ), 1 );
									break;
								case 'points':
									$score = $score / 10;
									break;
							}
						}

						$criterias[ $i ] = array(
							'title' => $title,
							'score' => $score,
						);

						if ( $criterias[ $i ] == array(
							'title' => '',
							'score' => '0',
						) ) {
							unset( $criterias[ $i ] );
						}

						if ( ! empty( $criterias ) ) {
							update_post_meta( $pid, '_lets_review_criterias', $criterias );
						}
					}

					switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
						case 'stars':
							update_post_meta( $pid, '_lets_review_format', 3 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', number_format( ( get_post_meta( $pid, 'cb_final_score_override', true ) / 20 ), 1 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', number_format( ( get_post_meta( $pid, 'cb_final_score', true ) / 20 ), 1 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
						case 'points':
							update_post_meta( $pid, '_lets_review_format', 2 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', ( get_post_meta( $pid, 'cb_final_score_override', true ) / 10 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', ( get_post_meta( $pid, 'cb_final_score', true ) / 10 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
						default:
							update_post_meta( $pid, '_lets_review_format', 1 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', get_post_meta( $pid, 'cb_final_score_override', true ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', get_post_meta( $pid, 'cb_final_score', true ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
					}
				}
				wp_reset_postdata();
				break;
			case 3:
				$qry = new WP_Query(
					array(
						'posts_per_page' => -1,
						'fields'         => 'ids',
						'meta_key'       => 'wppr_rating',
					)
				);
				if ( empty( $qry->posts ) ) {
					return;
				}

				$options = get_option( 'cwppos_options' );

				foreach ( $qry->posts as $pid ) {

					update_post_meta( $pid, 'cwp_meta_box_check', 'No', 'Yes' );
					update_post_meta( $pid, '_lets_review_onoff', 1, 0 );

					if ( get_post_meta( $pid, 'cwp_rev_product_name', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_title_use_post', '' );
						update_post_meta( $pid, '_lets_review_title', get_post_meta( $pid, 'cwp_rev_product_name', true ) );
					}

					if ( ! empty( $options['cwppos_pros_text'] ) ) {
						update_post_meta( $pid, '_lets_review_pros_title', $options['cwppos_pros_text'] );
					}

					if ( ! empty( $options['cwppos_cons_text'] ) ) {
						update_post_meta( $pid, '_lets_review_cons_title', $options['cwppos_cons_text'] );
					}

					$pros_new    = $cons_new = $criterias = $aff = array();
					$pro_options = get_post_meta( $pid, 'cwp_option_1_pro', true );
					if ( ! empty( $pro_options ) ) {

						for ( $i = 0; $i < 11; $i++ ) {
							$option_i_content = get_post_meta( $pid, 'cwp_option_' . $i . '_pro', true );
							if ( ! empty( $option_i_content ) ) {
								$pros_new[] = array( 'positive' => get_post_meta( $pid, 'cwp_option_' . $i . '_pro', true ) );
							}
						}
					} else {
						$wppr_pros = get_post_meta( $pid, 'wppr_pros', true );
						$pros_new  = array();
						if ( ! empty( $wppr_pros ) ) {
							foreach ( $wppr_pros as $key ) {
								$pros_new[] = array( 'positive' => $key );
							}
						}
					}

					if ( ! empty( $pros_new ) ) {
						update_post_meta( $pid, '_lets_review_pros', $pros_new );
					}
					$con_options = get_post_meta( $pid, 'cwp_option_1_cons', true );
					if ( ! empty( $con_options ) ) {
						for ( $i = 0; $i < 11; $i++ ) {
							$cwp_option = get_post_meta( $pid, 'cwp_option_' . $i . '_cons', true );
							if ( ! empty( $cwp_option ) ) {
								$cons_new[] = array( 'negative' => get_post_meta( $pid, 'cwp_option_' . $i . '_cons', true ) );
							}
						}
					} else {
						$wppr_cons = get_post_meta( $pid, 'wppr_cons', true );
						$cons_new  = array();
						if ( ! empty( $wppr_cons ) ) {
							foreach ( $wppr_cons as $key ) {
								$cons_new[] = array( 'negative' => $key );
							}
						}
					}

					if ( ! empty( $cons_new ) ) {
						update_post_meta( $pid, '_lets_review_cons', $cons_new );
					}
					$option_content = get_post_meta( $pid, 'option_1_content', true );
					if ( ! empty( $option_content ) ) {
						$f_score = 0;
						$n       = 0;
						switch ( get_option( 'lets_review_gd_format' ) ) {
							case 1:
								update_post_meta( $pid, '_lets_review_format', 1 );

								for ( $i = 0; $i < 11; $i++ ) {
									$option_i_content = get_post_meta( $pid, 'option_' . $i . '_content', true );
									if ( ! empty( $option_i_content ) ) {
										$criterias[] = array(
											'title' => get_post_meta( $pid, 'option_' . $i . '_content', true ),
											'score' => get_post_meta( $pid, 'option_' . $i . '_grade', true ),
										);
										$f_score     = $f_score + get_post_meta( $pid, 'option_' . $i . '_grade', true );
										$n++;
									}
								}

								update_post_meta( $pid, '_lets_review_final_score_100', intval( $f_score / $n ) );
								update_post_meta( $pid, '_lets_review_final_score', intval( $f_score / $n ) );

								break;
							case 2:
								update_post_meta( $pid, '_lets_review_format', 2 );
								$option_i_content = get_post_meta( $pid, 'option_' . $i . '_content', true );
								for ( $i = 0; $i < 11; $i++ ) {
									if ( ! empty( $option_i_content ) ) {
										$criterias[] = array(
											'title' => get_post_meta( $pid, 'option_' . $i . '_content', true ),
											'score' => get_post_meta( $pid, 'option_' . $i . '_grade', true ) / 10,
										);
										$f_score     = $f_score + get_post_meta( $pid, 'option_' . $i . '_grade', true );
										$n++;
									}
								}

								update_post_meta( $pid, '_lets_review_final_score_100', intval( $f_score / $n ) );
								update_post_meta( $pid, '_lets_review_final_score', ( $f_score / $n / 10 ) );

								break;
							default:
								update_post_meta( $pid, '_lets_review_format', 3 );

								for ( $i = 0; $i < 11; $i++ ) {
									$option_i_content = get_post_meta( $pid, 'option_' . $i . '_content', true );
									if ( ! empty( $option_i_content ) ) {
										$criterias[] = array(
											'title' => get_post_meta( $pid, 'option_' . $i . '_content', true ),
											'score' => number_format( ( get_post_meta( $pid, 'option_' . $i . '_grade', true ) / 20 ), 1 ),
										);
										$f_score     = $f_score + get_post_meta( $pid, 'option_' . $i . '_grade', true );
										$n++;
									}
								}

								update_post_meta( $pid, '_lets_review_final_score_100', intval( $f_score / $n ) );
								update_post_meta( $pid, '_lets_review_final_score', number_format( $f_score / $n / 20, 1 ) );

								break;
						}
					} else {
						$wppr_options = get_post_meta( $pid, 'wppr_options', true );
						if ( ! empty( $wppr_options ) ) {
							$f_score_base = intval( get_post_meta( $pid, 'wppr_rating', true ) );
							$f_score_100  = $f_score_base;
							switch ( get_option( 'lets_review_gd_format' ) ) {
								case 1:
									$format  = 1;
									$f_score = $f_score_base;
									break;
								case 2:
									$format  = 1;
									$f_score = $f_score_base / 10;
									break;
								default:
									$format  = 3;
									$f_score = number_format( ( $f_score_base / 20 ), 1 );
									break;
							}
							$criterias = array();
							foreach ( $wppr_options as $key ) {
								$criterias[] = array(
									'title' => $key['name'],
									'score' => $key['value'],
								);
							}

							update_post_meta( $pid, '_lets_review_format', $format );
							update_post_meta( $pid, '_lets_review_final_score_100', $f_score_100 );
							update_post_meta( $pid, '_lets_review_final_score', $f_score );
						}
					}
					$design = get_option( 'lets_review_gd_design' );
					update_post_meta( $pid, '_lets_review_design', $design );

					$format = get_option( 'lets_review_gd_format' );
					update_post_meta( $pid, '_lets_review_format', $format );

					$subtitle = get_option( 'lets_review_gd_title_score_subtitle' );
					update_post_meta( $pid, '_lets_review_subtitle', $subtitle );

					if ( ! empty( $criterias ) ) {
						update_post_meta( $pid, '_lets_review_criterias', $criterias );
					}
					$affiliate_text = get_post_meta( $pid, 'cwp_product_affiliate_text', true );
					if ( ! empty( $affiliate_text ) ) {

						$aff[] = array(
							'url'   => get_post_meta( $pid, 'cwp_product_affiliate_link', true ),
							'title' => get_post_meta( $pid, 'cwp_product_affiliate_text', true ),
						);

					}
					$affiliate_text2 = get_post_meta( $pid, 'cwp_product_affiliate_text2', true );
					if ( ! empty( $affiliate_text2 ) ) {

						$aff[] = array(
							'url'   => get_post_meta( $pid, 'cwp_product_affiliate_link2', true ),
							'title' => get_post_meta( $pid, 'cwp_product_affiliate_text2', true ),
						);

					}

					if ( empty( $aff ) ) {
						$wppr_links = get_post_meta( $pid, 'wppr_links', true );
						if ( ! empty( $wppr_links ) ) {
							foreach ( $wppr_links as $key => $value ) {
								$aff[] = array(
									'url'   => $value,
									'title' => $key,
								);
							}
						}
					}

					if ( ! empty( $aff ) ) {
						update_post_meta( $pid, '_lets_review_aff_buttons', $aff );
					}
					$product_image = get_post_meta( $pid, 'cwp_rev_product_image', true );
					if ( ! empty( $product_image ) ) {
						$fi = attachment_url_to_postid( get_post_meta( $pid, 'cwp_rev_product_image', true ) );
						update_post_meta( $pid, '_lets_review_fi', $fi );
					}
				}

				wp_reset_postdata();
				break;
			case 5:
				$qry = new WP_Query(
					array(
						'posts_per_page' => -1,
						'fields'         => 'ids',
					)
				);
				if ( empty( $qry->posts ) ) {
					return;
				}

				foreach ( $qry->posts as $pid ) {
					$criterias = (array) get_post_meta( $pid, '_post_review_box_breakdowns', true );
					if ( 1 === count( $criterias ) && empty( $criterias[0]['title'] ) && empty( $criterias[0]['rating'] ) ) {
						continue;
					}
					update_post_meta( $pid, '_lets_review_onoff', 1, 0 );
					update_post_meta( $pid, '_lets_review_format', 1 );
					update_post_meta( $pid, '_lets_review_title', get_post_meta( $pid, '_post_review_box_title', true ) );
					update_post_meta( $pid, '_lets_review_final_score', get_post_meta( $pid, '_post_review_box_breakdowns_score', true ) );
					update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, '_post_review_box_breakdowns_score', true ) );
					update_post_meta( $pid, '_lets_review_conclusion', get_post_meta( $pid, '_post_review_box_summary', true ) );
					update_post_meta( $pid, '_lets_review_conclusion_title', get_post_meta( $pid, '_post_review_box_summary_title', true ) );
					update_post_meta( $pid, '_lets_review_subtitle', get_post_meta( $pid, '_post_review_box_score_title', true ) );
					update_post_meta( $pid, '_lets_review_type', 2 );
					$criteria_output = array();
					foreach ( $criterias as $crit ) {
						$criteria_output[] = array(
							'title' => $crit['title'],
							'score' => $crit['rating'],
						);
					}

					if ( ! empty( $criteria_output ) ) {
						update_post_meta( $pid, '_lets_review_criterias', $criteria_output );
					}
				}
				wp_reset_postdata();
				break;
			case 4:
				$qry = new WP_Query(
					array(
						'posts_per_page' => -1,
						'fields'         => 'ids',
						'meta_key'       => 'cb_review_checkbox',
						'meta_value'     => 'on',
					)
				);
				if ( empty( $qry->posts ) ) {
					return;
				}

				foreach ( $qry->posts as $pid ) {

					delete_post_meta( $pid, 'cb_review_checkbox' );
					update_post_meta( $pid, '_lets_review_onoff', 1, 0 );

					if ( get_post_meta( $pid, 'cb_review_title_user', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_title', get_post_meta( $pid, 'cb_review_title_user', true ) );
					}

					if ( get_post_meta( $pid, 'cb_pros_title', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_pros_title', get_post_meta( $pid, 'cb_pros_title', true ) );
					}

					if ( get_post_meta( $pid, 'cb_cons_title', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_cons_title', get_post_meta( $pid, 'cb_cons_title', true ) );
					}

					if ( get_post_meta( $pid, 'cb_rating_short_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_subtitle_external', get_post_meta( $pid, 'cb_rating_short_summary', true ) );
					}

					if ( get_post_meta( $pid, 'cb_rating_short_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_subtitle', get_post_meta( $pid, 'cb_rating_short_summary', true ) );
					}

					if ( get_post_meta( $pid, 'cb_summary', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_conclusion', get_post_meta( $pid, 'cb_summary', true ) );
					}

					switch ( get_post_meta( $pid, 'cb_user_score', true ) ) {
						case 1:
							$type = 1;
							break;
						case 'on':
							$type = 1;
							break;
						case 'On':
							$type = 1;
							break;
						default:
							$type = 2;
							break;
					}

					update_post_meta( $pid, '_lets_review_type', $type );

					if ( get_post_meta( $pid, 'cb_placement', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_placement', true ) ) {
							case 'Top':
								$location = 5;
								break;
							default:
								$location = 2;
								break;
						}
						update_post_meta( $pid, '_lets_review_location', 2, 5 );
					}

					if ( get_post_meta( $pid, 'votes', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_user_rating_vote_count', get_post_meta( $pid, 'votes', true ) );
					}

					if ( get_post_meta( $pid, 'user_score', true ) != '' ) {
						update_post_meta( $pid, '_lets_review_user_rating', get_post_meta( $pid, 'user_score', true ) );
					}

					$pros_new = $cons_new = $criterias = array();

					if ( get_post_meta( $pid, 'cb_pros_one', true ) != '' ) {
						$pros_new[] = array( 'positive' => get_post_meta( $pid, 'cb_pros_one', true ) );
					}

					if ( get_post_meta( $pid, 'cb_pros_two', true ) != '' ) {
						$pros_new[] = array( 'positive' => get_post_meta( $pid, 'cb_pros_two', true ) );
					}

					if ( get_post_meta( $pid, 'cb_pros_three', true ) != '' ) {
						$pros_new[] = array( 'positive' => get_post_meta( $pid, 'cb_pros_three', true ) );
					}

					if ( ! empty( $pros_new ) ) {
						update_post_meta( $pid, '_lets_review_pros', $pros_new );
					}

					if ( get_post_meta( $pid, 'cb_cons_one', true ) != '' ) {
						$cons_new[] = array( 'negative' => get_post_meta( $pid, 'cb_cons_one', true ) );
					}
					if ( get_post_meta( $pid, 'cb_cons_two', true ) != '' ) {
						$cons_new[] = array( 'negative' => get_post_meta( $pid, 'cb_cons_two', true ) );
					}
					if ( get_post_meta( $pid, 'cb_cons_three', true ) != '' ) {
						$cons_new[] = array( 'negative' => get_post_meta( $pid, 'cb_cons_three', true ) );
					}

					if ( ! empty( $cons_new ) ) {
						update_post_meta( $pid, '_lets_review_cons', $cons_new );
					}

					if ( get_post_meta( $pid, 'cb_rating_1_title', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
							case 'Stars':
								$score = number_format( ( get_post_meta( $pid, 'cb_rating_1_score', true ) / 20 ), 1 );
								break;
							case 'Out of 10':
								$score = get_post_meta( $pid, 'cb_rating_1_score', true ) / 10;
								break;
							default:
								$score = get_post_meta( $pid, 'cb_rating_1_score', true );
								break;
						}
						$criterias[] = array(
							'title' => get_post_meta( $pid, 'cb_rating_1_title', true ),
							'score' => $score,
						);

					}

					if ( get_post_meta( $pid, 'cb_rating_2_title', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
							case 'Stars':
								$score = number_format( ( get_post_meta( $pid, 'cb_rating_2_score', true ) / 20 ), 1 );
								break;
							case 'Out of 10':
								$score = get_post_meta( $pid, 'cb_rating_2_score', true ) / 10;
								break;
							default:
								$score = get_post_meta( $pid, 'cb_rating_2_score', true );
								break;
						}
						$criterias[] = array(
							'title' => get_post_meta( $pid, 'cb_rating_2_title', true ),
							'score' => $score,
						);

					}

					if ( get_post_meta( $pid, 'cb_rating_3_title', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
							case 'Stars':
								$score = number_format( ( get_post_meta( $pid, 'cb_rating_3_score', true ) / 20 ), 1 );
								break;
							case 'Out of 10':
								$score = get_post_meta( $pid, 'cb_rating_3_score', true ) / 10;
								break;
							default:
								$score = get_post_meta( $pid, 'cb_rating_3_score', true );
								break;
						}
						$criterias[] = array(
							'title' => get_post_meta( $pid, 'cb_rating_3_title', true ),
							'score' => $score,
						);

					}

					if ( get_post_meta( $pid, 'cb_rating_4_title', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
							case 'Stars':
								$score = number_format( ( get_post_meta( $pid, 'cb_rating_4_score', true ) / 20 ), 1 );
								break;
							case 'Out of 10':
								$score = get_post_meta( $pid, 'cb_rating_4_score', true ) / 10;
								break;
							default:
								$score = get_post_meta( $pid, 'cb_rating_4_score', true );
								break;
						}
						$criterias[] = array(
							'title' => get_post_meta( $pid, 'cb_rating_4_title', true ),
							'score' => $score,
						);

					}

					if ( get_post_meta( $pid, 'cb_rating_5_title', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
							case 'Stars':
								$score = number_format( ( get_post_meta( $pid, 'cb_rating_5_score', true ) / 20 ), 1 );
								break;
							case 'Out of 10':
								$score = get_post_meta( $pid, 'cb_rating_5_score', true ) / 10;
								break;
							default:
								$score = get_post_meta( $pid, 'cb_rating_5_score', true );
								break;
						}
						$criterias[] = array(
							'title' => get_post_meta( $pid, 'cb_rating_5_title', true ),
							'score' => $score,
						);

					}

					if ( get_post_meta( $pid, 'cb_rating_6_title', true ) != '' ) {

						switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
							case 'Stars':
								$score = number_format( ( get_post_meta( $pid, 'cb_rating_6_score', true ) / 20 ), 1 );
								break;
							case 'Out of 10':
								$score = get_post_meta( $pid, 'cb_rating_6_score', true ) / 10;
								break;
							default:
								$score = get_post_meta( $pid, 'cb_rating_6_score', true );
								break;
						}
						$criterias[] = array(
							'title' => get_post_meta( $pid, 'cb_rating_6_title', true ),
							'score' => $score,
						);

					}

					if ( ! empty( $criterias ) ) {
						update_post_meta( $pid, '_lets_review_criterias', $criterias );
					}

					switch ( get_post_meta( $pid, 'cb_score_display_type', true ) ) {
						case 'Stars':
							update_post_meta( $pid, '_lets_review_format', 3 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', number_format( ( get_post_meta( $pid, 'cb_final_score_override', true ) / 20 ), 1 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', number_format( ( get_post_meta( $pid, 'cb_final_score', true ) / 20 ), 1 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
						case 'Out of 10':
							update_post_meta( $pid, '_lets_review_format', 2 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', ( get_post_meta( $pid, 'cb_final_score_override', true ) / 10 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', ( get_post_meta( $pid, 'cb_final_score', true ) / 10 ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score', true ) );
							}

							break;
						default:
							update_post_meta( $pid, '_lets_review_format', 1 );

							if ( get_post_meta( $pid, 'cb_final_score_override', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', intval( get_post_meta( $pid, 'cb_final_score_override', true ) ) );
								update_post_meta( $pid, '_lets_review_final_score_100', get_post_meta( $pid, 'cb_final_score_override', true ) );
							} elseif ( get_post_meta( $pid, 'cb_final_score', true ) != '' ) {
								update_post_meta( $pid, '_lets_review_final_score', intval( get_post_meta( $pid, 'cb_final_score', true ) ) );
								update_post_meta( $pid, '_lets_review_final_score_100', intval( get_post_meta( $pid, 'cb_final_score', true ) ) );
							}

							break;
					}
				}

				wp_reset_postdata();
				break;

		}

	}

	/**
	 * Admin Page Head
	 *
	 * @since    1.0.0
	 */
	public function lets_review_admin_head() {
		?>
		<div class="wrap about-wrap lets-review-wrap">
		<h1><?php echo esc_html( $this->name ) . ' ' . esc_html( $this->version ); ?></h1>
		<div class="about-text"><?php esc_html_e( 'Thank you for choosing Let\'s Review and for joining the Codetipi community. Happy reviewing!', 'lets-review' ); ?></div>
		<div class="lets-review-badge"><?php echo esc_html( $this->version ); ?></div>
		<div class="lets-review-badge"><img src="<?php echo esc_url( $this->url . 'assets/admin/img/badge.jpg' ); ?>" srcset="<?php echo esc_url( $this->url . 'assets/admin/img/badge@2x.jpg' ); ?> 2x" alt=""><span class="version"><?php echo esc_html( $this->version ); ?></span></div>

		<h2 class="nav-tab-wrapper">
		<?php
		$tabs = array(
			'lets-review'           => esc_html__( 'Welcome', 'lets-review' ),
			'lets-review-general'   => esc_html__( 'Options', 'lets-review' ),
			'lets-review-default'   => esc_html__( 'Default Values', 'lets-review' ),
			'lets-review-help'      => esc_html__( 'Help', 'lets-review' ),
			'lets-review-migration' => esc_html__( 'Migration', 'lets-review' ),
		);
		?>
		<?php foreach ( $tabs as $tab => $name ) { ?>
			<?php $class = $tab == $_GET['page'] ? 'nav-tab-active' : ''; ?>
			<?php if ( 'lets-review-theme-options' == $tab ) { ?>
				<a class="nav-tab <?php echo esc_attr( $class ); ?>" href="customize.php?return=admin.php?page=lets-review"><?php echo esc_html( $name ); ?></a>
			<?php } else { ?>
				<a class="nav-tab <?php echo esc_attr( $class ); ?>" href="?page=<?php echo esc_attr( $tab ); ?>"><?php echo esc_html( $name ); ?></a>
			<?php } ?>

		<?php } ?>
		</h2>

		<?php
	}

	/**
	 * Admin Page Footer
	 *
	 * @since    1.0.0
	 */
	public function lets_review_admin_footer() {
		?>
	<div class="tipi-box tipi-xs-12 tipi-m-6 tipi-col tipi-col-first">
		<a href="http://facebook.com/codetipi" target="_blank"><img src="<?php echo esc_url( plugin_dir_url( dirname( __FILE__ ) ) . 'admin/images/tipi-fb.png' ); ?>" alt=""></a>
	</div>
	<div class="tipi-box tipi-xs-12 tipi-m-6 tipi-col tipi-col-last">
		<a href="http://twitter.com/codetipi" target="_blank"><img src="<?php echo esc_url( plugin_dir_url( dirname( __FILE__ ) ) . 'admin/images/tipi-tw.png' ); ?>" alt=""></a>
	</div>
	</div>
		<?php
	}

}
